/*==========================================================================
  REGMAP.C

  Example code to write and read a memory mapped register using the VGA
  aperture, 4M linear aperture size, or 8M linear aperture size. For more
  details, see functions REGW() and REGR() in MEMREG.C.

  Note that for GUI register access with the linear aperture, the aperture
  address must be valid. The INSTALL program should be used to ENABLE the
  linear aperture.

  Possible failures:

    - VGA aperture register access will fail if the VGA controller is
      disabled

    - Linear aperture register access will fail if the memory mapped
      register area is overlapping extended memory or resides at an invalid
      address


  Copyright (c) 1994-1995 ATI Technologies Inc. All rights reserved
 =========================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include "..\util\atim64.h"
#include "..\util\sample.h"

#define INSTANCE 0

int main (void)
{
    int config;
    unsigned long regval;

    // check if Mach64 adapter is installed
    if (detect_mach64 (INSTANCE) != YES_MACH64)
    {
        printf ("mach64 based adapter was not found.\n");
        return (1);
    }

    // fill global query structure by calling Mach 64 ROM
    if (query_hardware () != NO_ERROR)
    {
        printf ("Failed ROM call to query mach64 hardware.\n");
        return (1);
    }

    // save config value
    config = ior16 (ioCONFIG_CNTL);

    // Enable GUI engine to turn on GUI memory mapped register access
    reset_engine ();

    // Insure that fifo is empty before starting - enable linear aperture
    // to insure that the wait_for_idle() can read the FIFO_STAT and
    // GUI_STAT registers. The VGA aperture may not be available. This is
    // necessary since the register under test (PAT_REG0) is a GUI memory
    // mapped register.

    if (init_aperture () == APERTURE_ENABLE_FAILURE)
    {
        printf ("Could not enable an aperture.\n");
        // restore configuration
        iow16 (ioCONFIG_CNTL, config);
        return (1);
    }
    update_aperture_status ();
    wait_for_idle ();

    // --------------------------------------------------------------------
    // Use the VGA aperture to access the GUI registers. The base address
    // is B000:FC00h. Each register occupies 4 bytes.

    // Use VGA aperture if the VGA controller is enabled
    if (querydata.vga_type == VGA_ENABLE)
    {
        // enable the VGA aperture for register access
        disable_linear_aperture ();
        if (enable_vga_aperture () == APERTURE_VGA_ENABLE_SUCCESS)
        {
            // Use this function to insure regw() and regr() work correctly
            update_aperture_status ();

            // write and read the PAT_REG0 register using the VGA aperture
            regval = 0x555AA5AA;
            printf ("VGA aperture enabled      : "
                    "write value = %08lX, read value = ", regval);

            regw (PAT_REG0, regval);

            regval = regr (PAT_REG0);
            printf ("%08lX\n", regval);
        }
        else
        {
            printf ("Could not enable VGA aperture.\n");
        }
    }
    else
    {
        printf ("VGA controller is disabled.\n");
    }

    // --------------------------------------------------------------------
    // Use the 4Meg linear aperture size to access the GUI registers. The
    // base address is the aperture address + 0x3FFC00h. Each register
    // occupies
    // 4 bytes.

    // enable the 4M linear memory aperture without distribing address
    disable_vga_aperture ();
    if (enable_linear_aperture (APERTURE_4M_ENABLE) ==
        APERTURE_4M_ENABLE_SUCCESS)
    {
        // Use this function to insure regw() and regr() work correctly
        update_aperture_status ();

        // write and read the PAT_REG0 register using the 4Meg linear aperture
        regval = 0xAAA55A55;
        printf ("4M Linear aperture enabled: "
                "write value = %08lX, read value = ", regval);

        regw (PAT_REG0, regval);

        regval = regr (PAT_REG0);
        printf ("%08lX\n", regval);
    }
    else
    {
        printf ("Could not enable 4MB linear aperture.\n");
    }

    // --------------------------------------------------------------------
    // Use the 8Meg linear aperture size to access the GUI registers. The
    // base address is the aperture address + 0x7FFC00h. Each register
    // occupies 4 bytes.

    // enable the 8M linear memory aperture without distribing address
    if (enable_linear_aperture (APERTURE_8M_ENABLE) ==
        APERTURE_8M_ENABLE_SUCCESS)
    {
        // Use this function to insure regw() and regr() work correctly
        update_aperture_status ();

        // write and read the PAT_REG0 register using the 8Meg linear aperture
        regval = 0x5A55AAA5;
        printf ("8M Linear aperture enabled: "
                "write value = %08lX, read value = ", regval);

        regw (PAT_REG0, regval);

        regval = regr (PAT_REG0);
        printf ("%08lX\n", regval);
    }
    else
    {
        printf ("Could not enable 8MB linear aperture.\n");
    }

    // restore configuration
    iow16 (ioCONFIG_CNTL, config);

    return (0);
}


