; ===========================================================================
; MOVEMEM.ASM
;
; Routine to move a block of data from or to extended memory. This routine
; will typically be used to write and read from the MACH64 aperture memory.
;
; Compiling:
;   masm /Ml /D<memory model> movemem.asm;
;       <memory model> = mem_S for SMALL model,
;                        mem_M for MEDIUM model,
;                        mem_L for LARGE model
;
; Copyright (c) 1993-1995 ATI Technologies Inc.  All rights reserved
; ===========================================================================

IFDEF mem_S
PARM        equ     4   ; passed parameters start at bp+4 for small model
ELSE
PARM        equ     6   ; passed parameters start at bp+6 for other models
ENDIF

IFDEF mem_S
.MODEL  SMALL, C
ELSEIFDEF mem_M
.MODEL  MEDIUM, C
ELSE
.MODEL  LARGE, C
ENDIF

.DATA

EVEN
                ; gdt structure required for INT 15h, function 87h call
gdt             dw      0,0,0,0
                dw      0,0,0,0
source_len      dw      0FFFFh              ; set to 0FFFFh
source_addr_low dw      ?                   ; 24 bit address
source_addr_hi  db      ?
source_access   db      93h                 ; set to 93h
source_addr_ext dw      0                   ; 386/486 address extensions
target_len      dw      0FFFFh              ; set to 0FFFFh
target_addr_low dw      ?                   ; 24 bit address
target_addr_hi  db      ?
target_access   db      93h                 ; set to 93h
target_addr_ext dw      0                   ; 386/486 address extensions
                dw      0,0,0,0
                dw      0,0,0,0

.CODE
.286

; Macro for 'call' model handling
Mcall       macro   routine
IFDEF mem_S
            call    NEAR PTR routine
ELSE
            call    FAR PTR routine
ENDIF
            endm

; ---------------------------------------------------------------------------
; MOVEMEM - Move a block of data to or from an extended address (at or above
;           the 1Meg memory boundary). Int 15h, function 87h is used for this
;           function. The input parameters are fetched from the stack.
;
; Inputs : FAR PTR: memory buffer address (segment:offset),
;          DWORD  : extended physical memory address,
;          WORD   : number of words to transfer (<= 32767 words),
;          WORD   : transfer direction flag
;                     0 -> READ  (extended memory to memory buffer)
;                     1 -> WRITE (memory buffer to extended memory)
;
; Outputs: Return value from Int 15h, function 87h in ax
;              0 for success,
;             >0 for error code
;
; C declaration for this function:
;
;     int movemem (void far *ptr,
;                  unsigned long dest,
;                  unsigned int nwords,
;                  int direction);
;
; The memory buffer address pointer is expected in segment:offset format. The
; 'dest' parameter is expected in physical address format.
; ---------------------------------------------------------------------------
            public  movemem

IFDEF mem_S
movemem     proc    near
ELSE
movemem     proc    far
ENDIF
            ; create frame pointer
            push    bp
            mov     bp, sp

            ; save registers used
            push    bx
            push    cx
            push    dx
            push    di
            push    si
            push    es

            ; retrieve direction flag in di
            mov     di, WORD PTR [bp+PARM+10]

            ; setup gdt structure
            mov     source_len, 0FFFFh
            mov     target_len, 0FFFFh
            mov     source_access, 93h
            mov     target_access, 93h

            ; calculate physical address from PTR address for GDT in dx:cx
            mov     dx, WORD PTR [bp+PARM+2] ; segment of PTR address
            mov     cx, dx
            clc
            shr     dx, 0Ch
            shl     cx, 4
            mov     ax, WORD PTR [bp+PARM]   ; load offset of PTR address
            add     cx, ax
            jnc     mm1
            inc     dx
mm1:
            ; put extended physical memory address in bx:ax
            mov     ax, WORD PTR [bp+PARM+4] ; load low word of address
            mov     bx, WORD PTR [bp+PARM+6] ; load high word of address

            ; fill in address depending on direction flag
            cmp     di, 0                    ; 0 = read
            je      mm2

            ; WRITE: source = mem buffer, target = extended mem address
            mov     source_addr_low, cx
            mov     source_addr_hi, dl
            xor     dl, dl
            mov     source_addr_ext, dx
            mov     target_addr_low, ax
            mov     target_addr_hi, bl
            xor     bl, bl
            mov     target_addr_ext, bx
            jmp     mm3
mm2:
            ; READ: source = extended mem address, target = mem buffer
            mov     source_addr_low, ax
            mov     source_addr_hi, bl
            xor     bl, bl
            mov     source_addr_ext, bx
            mov     target_addr_low, cx
            mov     target_addr_hi, dl
            xor     dl, dl
            mov     target_addr_ext, dx
mm3:
            ; setup registers and call INT 15h, function 87h, gdt = es:si
            mov     cx, WORD PTR [bp+PARM+8] ; load word transfer count
            mov     si, offset ds:gdt
            push    ds
            pop     es
            clc
            mov     ah, 87h
            int     15h

            ; setup return value
            mov     al, ah
            xor     ah, ah

            ; restore saved registers
            pop     es
            pop     si
            pop     di
            pop     dx
            pop     cx
            pop     bx

            ; remove frame pointer
            mov     sp, bp
            pop     bp

            ret

movemem     endp

            end

